# Design Document: LCPFF Website Redesign and Application Portal

## Overview

The LCPFF website redesign creates a modern, secure digital platform serving dual purposes: a marketing showcase for South African film financing and a comprehensive application portal for production companies. The system employs a component-based architecture with React.js frontend, Node.js backend, and cloud-native infrastructure to deliver a responsive, accessible, and secure user experience.

The design prioritizes user experience through a mobile-first approach, accommodating the 40% mobile traffic expectation while maintaining professional aesthetics aligned with LCPFF's brand identity. Security and compliance with POPIA regulations are foundational elements, implemented through encryption, secure session management, and comprehensive audit trails.

## Architecture

### System Architecture

The application follows a modern three-tier architecture pattern:

```mermaid
graph TB
    subgraph "Client Tier"
        A[React SPA] --> B[PWA Service Worker]
        A --> C[State Management - Redux Toolkit]
    end
    
    subgraph "Application Tier"
        D[Node.js API Gateway] --> E[Authentication Service]
        D --> F[Application Service]
        D --> G[File Upload Service]
        D --> H[Notification Service]
    end
    
    subgraph "Data Tier"
        I[PostgreSQL Database] --> J[Application Data]
        I --> K[User Profiles]
        L[AWS S3] --> M[File Storage]
        N[Redis Cache] --> O[Session Store]
    end
    
    A --> D
    E --> I
    F --> I
    G --> L
    H --> P[Email Service]
```

### Technology Stack

**Frontend Framework**: React.js 18+ with TypeScript
- Component-based architecture for reusability and maintainability
- React Router for client-side routing and navigation
- Redux Toolkit for predictable state management
- React Hook Form for efficient form handling with validation
- Material-UI (MUI) customized to match LCPFF brand guidelines

**Backend Services**: Node.js with Express.js
- RESTful API design following OpenAPI 3.0 specification
- JWT-based authentication with refresh token rotation
- Microservices architecture for scalability and maintainability
- Rate limiting and request throttling for security

**Database Layer**: PostgreSQL 14+
- ACID compliance for financial data integrity
- Row-level security for multi-tenant data isolation
- Full-text search capabilities for project showcase
- Automated backup and point-in-time recovery

**File Storage**: AWS S3 with CloudFront CDN
- Server-side encryption with AWS KMS
- Presigned URLs for secure direct uploads
- Automatic virus scanning integration
- Lifecycle policies for cost optimization

**Infrastructure**: AWS Cloud Services
- Application Load Balancer for high availability
- Auto Scaling Groups for elastic capacity
- AWS WAF for application-layer security
- CloudWatch for monitoring and alerting

## Components and Interfaces

### Frontend Components

**Layout Components**:
- `Header`: Navigation, user authentication status, brand identity
- `Footer`: Links, contact information, compliance statements
- `Sidebar`: Dashboard navigation, application progress indicators
- `Layout`: Responsive container managing mobile/desktop layouts

**Authentication Components**:
- `LoginForm`: Email/password authentication with validation
- `RegisterForm`: Company registration with CIPC verification
- `PasswordReset`: Secure password recovery workflow
- `EmailVerification`: Account activation process

**Application Components**:
- `ApplicationWizard`: Six-step form progression with auto-save
- `StepIndicator`: Visual progress tracking across application steps
- `FileUploader`: Drag-and-drop interface with progress indicators
- `DocumentViewer`: PDF preview and download functionality
- `DigitalSignature`: Electronic signature capture and validation

**Dashboard Components**:
- `DashboardHome`: Overview of applications and company profile
- `ApplicationList`: Tabular view with filtering and sorting
- `StatusPipeline`: Visual representation of application progress
- `ProfileEditor`: Company information management interface

**Showcase Components**:
- `ProjectGrid`: Responsive gallery with lazy loading
- `ProjectCard`: Individual project preview with metadata
- `ProjectDetail`: Comprehensive project information display
- `FilterPanel`: Search and categorization controls

### API Interfaces

**Authentication Endpoints**:
```typescript
POST /api/auth/register
POST /api/auth/login
POST /api/auth/refresh
POST /api/auth/logout
POST /api/auth/verify-email
POST /api/auth/reset-password
```

**Application Management**:
```typescript
GET /api/applications
POST /api/applications
GET /api/applications/:id
PUT /api/applications/:id
DELETE /api/applications/:id/draft
POST /api/applications/:id/submit
```

**File Management**:
```typescript
POST /api/files/upload-url
POST /api/files/complete-upload
GET /api/files/:id
DELETE /api/files/:id
```

**User Profile**:
```typescript
GET /api/profile
PUT /api/profile
GET /api/profile/company
PUT /api/profile/company
```

### Database Schema

**Core Tables**:
- `users`: Authentication and basic user information
- `companies`: Production company profiles and CIPC data
- `applications`: Application submissions and metadata
- `application_steps`: Individual step data and validation status
- `files`: File metadata and storage references
- `projects`: Funded project showcase information
- `audit_logs`: Comprehensive activity tracking for compliance

**Security Tables**:
- `user_sessions`: Active session management
- `login_attempts`: Brute force protection tracking
- `password_resets`: Secure token management
- `email_verifications`: Account activation tracking

## Data Models

### User and Company Models

```typescript
interface User {
  id: string;
  email: string;
  passwordHash: string;
  emailVerified: boolean;
  companyId: string;
  role: 'applicant' | 'admin';
  createdAt: Date;
  lastLoginAt: Date;
}

interface Company {
  id: string;
  name: string;
  registrationNumber: string;
  cipcVerified: boolean;
  contactPerson: string;
  phoneNumber: string;
  address: Address;
  bankingDetails: BankingDetails;
  createdAt: Date;
  updatedAt: Date;
}
```

### Application Models

```typescript
interface Application {
  id: string;
  referenceNumber: string;
  companyId: string;
  projectTitle: string;
  status: ApplicationStatus;
  currentStep: number;
  submittedAt?: Date;
  reviewedAt?: Date;
  approvedAt?: Date;
  steps: ApplicationStep[];
  files: FileReference[];
  signatures: DigitalSignature[];
}

interface ApplicationStep {
  stepNumber: number;
  stepName: string;
  data: Record<string, any>;
  completed: boolean;
  validationErrors: ValidationError[];
  lastModified: Date;
}

type ApplicationStatus = 
  | 'draft'
  | 'submitted'
  | 'administrative_review'
  | 'investment_committee'
  | 'conditional_approval'
  | 'legal_contracting'
  | 'funds_disbursed'
  | 'rejected';
```

### File and Security Models

```typescript
interface FileReference {
  id: string;
  applicationId: string;
  filename: string;
  originalName: string;
  mimeType: string;
  size: number;
  s3Key: string;
  uploadedAt: Date;
  virusScanned: boolean;
  encryptionKey: string;
}

interface AuditLog {
  id: string;
  userId: string;
  action: string;
  resourceType: string;
  resourceId: string;
  ipAddress: string;
  userAgent: string;
  timestamp: Date;
  metadata: Record<string, any>;
}
```

## Correctness Properties

*A property is a characteristic or behavior that should hold true across all valid executions of a system—essentially, a formal statement about what the system should do. Properties serve as the bridge between human-readable specifications and machine-verifiable correctness guarantees.*

### Property 1: Input Validation Consistency
*For any* user registration attempt, the system should validate email format and password strength according to defined rules, rejecting invalid inputs and accepting valid ones consistently
**Validates: Requirements 1.2**

### Property 2: CIPC Verification Integration
*For any* valid CIPC registration number provided during registration, the system should successfully verify the company exists in the registry and store the verification status
**Validates: Requirements 1.3**

### Property 3: Email Notification Delivery
*For any* application status change, account registration, or system event requiring notification, the system should send appropriate emails within the specified timeframe using consistent branding and templates
**Validates: Requirements 1.4, 10.1, 10.2, 10.3, 10.5**

### Property 4: Session Security Management
*For any* authenticated user session, the system should enforce automatic timeout after 30 minutes of inactivity and implement secure session management with proper cleanup
**Validates: Requirements 1.6, 9.3**

### Property 5: Auto-save Persistence
*For any* form field modification in the application wizard, the system should automatically save draft data within 5 seconds and preserve it across navigation and session boundaries
**Validates: Requirements 2.2, 2.3**

### Property 6: File Upload Validation and Security
*For any* file upload attempt, the system should validate file type and size constraints, encrypt accepted files, generate unique identifiers, and provide secure storage with proper metadata tracking
**Validates: Requirements 2.4, 3.1, 3.2, 3.3, 3.4, 3.5**

### Property 7: Application Reference Uniqueness
*For any* completed application submission, the system should generate a unique reference number that remains consistent and identifiable throughout the application lifecycle
**Validates: Requirements 2.6**

### Property 8: Application Status Pipeline Integrity
*For any* application status change, the system should update the visual pipeline correctly, maintain chronological progression, and provide accurate status information with timestamps
**Validates: Requirements 5.1, 5.2, 5.3, 5.4, 5.5**

### Property 9: Project Showcase Data Consistency
*For any* project displayed in the showcase, the system should present complete metadata including budget, genre, production company, and status information consistently across list and detail views
**Validates: Requirements 6.2, 6.4**

### Property 10: Search and Filter Functionality
*For any* search query or filter criteria applied to projects or resources, the system should return relevant results with proper highlighting and maintain filter state consistency
**Validates: Requirements 6.4, 7.3**

### Property 11: Performance Optimization
*For any* page load or resource request, the system should complete initial rendering within 3 seconds and implement lazy loading for optimal performance across different connection speeds
**Validates: Requirements 6.5, 9.1**

### Property 12: Responsive Design Adaptation
*For any* viewport size change or device orientation, the system should adapt layouts responsively without horizontal scrolling and maintain touch-friendly navigation on mobile devices
**Validates: Requirements 8.1, 8.2**

### Property 13: Accessibility Compliance
*For any* user interaction method (keyboard navigation, screen readers, high contrast needs), the system should provide appropriate accessibility features meeting WCAG 2.1 AA standards
**Validates: Requirements 8.3, 8.4, 8.5**

### Property 14: Data Security and Encryption
*For any* data transmission or storage operation, the system should enforce SSL encryption, implement proper data protection measures, and comply with POPIA requirements for personal information handling
**Validates: Requirements 9.2, 9.4, 9.5**

### Property 15: Notification Preference Management
*For any* user notification preference change, the system should respect opt-out settings while maintaining delivery of critical communications required for application processing
**Validates: Requirements 10.4**

### Property 16: Content Version Control
*For any* content modification in the CMS, the system should maintain version history, allow rollback capabilities, and automatically notify relevant users of changes
**Validates: Requirements 11.2, 11.4**

### Property 17: Digital Signature Integrity
*For any* document requiring digital signature, the system should capture legally binding signatures with complete audit trails, generate signed PDF copies, and coordinate multi-party signing processes
**Validates: Requirements 12.2, 12.3, 12.4, 12.5**

## Error Handling

### Client-Side Error Handling

**Form Validation Errors**:
- Real-time field validation with immediate user feedback
- Comprehensive error messages with specific correction guidance
- Prevention of form submission until all validation errors are resolved
- Graceful handling of network connectivity issues during auto-save

**File Upload Errors**:
- Clear error messages for unsupported file types or oversized files
- Progress indicators with cancellation options for large uploads
- Retry mechanisms for failed uploads with exponential backoff
- Virus scan failure notifications with alternative submission methods

**Authentication Errors**:
- Account lockout protection after multiple failed login attempts
- Clear messaging for expired sessions with automatic redirect to login
- Password reset flow with secure token validation
- Email verification reminders for unactivated accounts

### Server-Side Error Handling

**Database Errors**:
- Transaction rollback for data integrity failures
- Connection pool management with automatic retry logic
- Graceful degradation for read-only operations during maintenance
- Comprehensive logging for debugging and monitoring

**External Service Failures**:
- Circuit breaker pattern for CIPC API integration
- Fallback mechanisms for email delivery failures
- Timeout handling for file storage operations
- Rate limiting protection against API abuse

**Security Violations**:
- Automatic session termination for suspicious activity
- IP-based blocking for repeated security violations
- Comprehensive audit logging for compliance requirements
- Secure error messages that don't expose system internals

### Error Recovery Strategies

**Data Recovery**:
- Automatic draft recovery from browser local storage
- Database backup and point-in-time recovery procedures
- File corruption detection and alternative retrieval methods
- User notification systems for data loss prevention

**System Recovery**:
- Health check endpoints for monitoring system status
- Automated failover for critical system components
- Graceful degradation modes for non-essential features
- Maintenance mode with user communication

## Testing Strategy

### Dual Testing Approach

The LCPFF application requires comprehensive testing through both unit tests and property-based tests to ensure correctness, security, and compliance. Unit tests validate specific examples and edge cases, while property tests verify universal properties across all inputs. Together, they provide comprehensive coverage where unit tests catch concrete bugs and property tests verify general correctness.

### Unit Testing Strategy

**Component Testing**:
- React component rendering with various props and state combinations
- User interaction simulation (clicks, form submissions, navigation)
- Error boundary testing for graceful failure handling
- Accessibility testing with screen reader simulation

**API Testing**:
- Endpoint response validation for various input scenarios
- Authentication and authorization flow testing
- File upload and download functionality verification
- Database transaction integrity testing

**Integration Testing**:
- End-to-end application workflow testing
- External service integration (CIPC API, email services)
- Cross-browser compatibility verification
- Mobile device responsiveness testing

### Property-Based Testing Configuration

**Testing Framework**: Jest with fast-check library for JavaScript/TypeScript
- Minimum 100 iterations per property test for comprehensive input coverage
- Each property test references its corresponding design document property
- Tag format: **Feature: lcpff-website-redesign, Property {number}: {property_text}**

**Property Test Implementation**:
- Input validation properties using generated email addresses, passwords, and form data
- File upload properties with various file types, sizes, and content
- Session management properties with different timeout scenarios
- Status pipeline properties with various application states and transitions
- Security properties with different authentication and authorization scenarios

**Test Data Generation**:
- Realistic company information generation for registration testing
- Valid and invalid CIPC registration numbers for verification testing
- Various file types and sizes for upload validation testing
- Different user roles and permissions for authorization testing
- Multiple application states for status pipeline testing

### Performance Testing

**Load Testing**:
- Concurrent user simulation for application submission workflows
- File upload performance under various network conditions
- Database query optimization validation
- CDN and caching effectiveness measurement

**Security Testing**:
- Penetration testing for common web vulnerabilities (OWASP Top 10)
- Authentication bypass attempt simulation
- File upload security validation (malware, oversized files)
- SQL injection and XSS vulnerability testing
- POPIA compliance validation for data handling procedures

### Compliance Testing

**Accessibility Testing**:
- WCAG 2.1 AA compliance validation using automated tools
- Manual testing with screen readers and keyboard navigation
- Color contrast ratio verification across all UI components
- Mobile accessibility testing for touch interfaces

**Security Compliance**:
- POPIA compliance audit for personal data handling
- SSL/TLS configuration validation
- Session security and timeout verification
- Audit trail completeness and integrity testing

Each correctness property must be implemented as a single property-based test with appropriate generators for test data. The testing strategy ensures that the LCPFF application meets all functional, security, and compliance requirements while providing a robust and reliable user experience for South African film production companies.